"use client";
import Image from "next/image";
import React, { useState } from "react";
import { useFormik } from "formik";
import * as Yup from "yup";
import bgimg from "../../../../public/assets/login/bgimg.png";
import image from "../../../../public/assets/foget-password/image.png";
import Link from "next/link";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faEye, faEyeSlash } from "@fortawesome/free-solid-svg-icons";
import toast from "react-hot-toast";
import { useRouter } from "next/navigation";
import Cookies from "js-cookie";
import { useRegisterUserMutation } from "@/services/Api";
import { useDispatch } from "react-redux";
import { loggedIn } from "@/redux/authSlice";

const Signup = () => {
  const router = useRouter();
  const [passwordVisible, setPasswordVisible] = useState(false);
  const [confirmVisible, setConfirmVisible] = useState(false);
  const [error, setError] = useState("");
  const dispatch = useDispatch();

  const [registerUser] = useRegisterUserMutation();

  const formik = useFormik({
    initialValues: {
      name: "",
      email: "",
      password: "",
      password_confirmation: "",
    },

    validationSchema: Yup.object({
      name: Yup.string()
        .min(2, "Too Short!")
        .max(50, "Too Long!")
        .required("Name is Required"),
      email: Yup.string()
        .email("Invalid email address")
        .required("Email is required"),
      password: Yup.string()
        .required("Password is required")
        .min(8, "Password is too short - should be 8 chars minimum.")
        .matches(/[a-zA-Z]/, "Password can only contain Latin letters.")
        .matches(
          /[!@#$%^&*(),.?":{}|<>]/,
          "Password must contain at least one special character."
        ),
      password_confirmation: Yup.string()
        .oneOf([Yup.ref("password"), null], "Passwords must match")
        .required("Confirm Password is required"),
    }),

    onSubmit: async (values, { setSubmitting }) => {
      try {
        const res = await registerUser({ data: values });
        if (
          res?.error?.data?.errors?.email[0] === "The email has already been taken."
        ) {
          toast.error("The email has already been taken.", {
            position: "top-center",
          });
        } else {
          const token = res?.data?.token;
          const role = res?.data?.roles[0];
          localStorage.setItem("authToken", token);
          const userDetail = {
            id: res?.data?.user?.id,
            name: res?.data?.user?.name,
            role: res?.data?.user?.roles[0],
          };
          const response = {
            token: res?.data?.token,
            user: userDetail,
          };
          dispatch(loggedIn(response));
          
          Cookies.set("authToken", token, { path: "/otp", expires: 7 });
          Cookies.set("role", role, { path: "/otp", expires: 7 });
          router.push("/otp");
          toast.success("Signup successful!", {
            position: "top-center",
          });
        }
      } catch (error) {
        console.error("signUp error:", error.response?.data);
      } finally {
        setSubmitting(false);
      }
    },
  });

  return (
    <div className="w-full max-h-[900px] flex justify-between relative">
      <div className="absolute z-10 h-full lg:left-[20%] md:left-[10%] sm:left-[25%] left-[15%] sm:min-w-[383px] w-[300px]">
        <div className="flex flex-col justify-around h-full w-full">
          <div className="flex justify-center">
            <Image
              src="/assets/navbar/logo1.svg"
              alt="logo"
              className="ml-14 w-40 h-40 lg:w-52 lg:h-40 mb-6 md:mb-4"
              width={100}
              height={100}
            />
          </div>
          <form
            className="flex flex-col md:gap-4 gap-4"
            onSubmit={formik.handleSubmit}
          >
            <div className="flex flex-col md:gap-4 gap-3 md:bg-transparent bg-whitegrey p-5 rounded-2xl">
              <div className="flex flex-col gap-3">
                <h1 className="text-black text-3xl font-bold text-center">
                  Registration
                </h1>
              </div>
              <div className="flex flex-col gap-2">
                <label>Name:</label>
                <input
                  type="text"
                  className={`outline-none py-2 px-3 border border-grey rounded-xl ${
                    formik.touched.name && formik.errors.name
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                  placeholder="Enter your Name"
                  name="name"
                  value={formik.values.name}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                />
                {formik.touched.name && formik.errors.name ? (
                  <div className="text-red-500">{formik.errors.name}</div>
                ) : null}
              </div>
              <div className="flex flex-col gap-2">
                <label>Email:</label>
                <input
                  type="email"
                  className={`outline-none py-2 px-3 border border-grey rounded-xl ${
                    formik.touched.email && formik.errors.email
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                  placeholder="Enter your email address"
                  name="email"
                  value={formik.values.email}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                />
                {formik.touched.email && formik.errors.email ? (
                  <div className="text-red-500">{formik.errors.email}</div>
                ) : null}
              </div>
              <div className="relative flex flex-col gap-2">
                <label>Password:</label>
                <input
                  type={passwordVisible ? "text" : "password"}
                  className={`outline-none py-2 px-3 border border-grey rounded-xl ${
                    formik.touched.password && formik.errors.password
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                  placeholder="Enter New Password"
                  name="password"
                  value={formik.values.password}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                />
                <button
                  type="button"
                  onClick={() => setPasswordVisible(!passwordVisible)}
                  className="absolute text-darkgrey right-3 top-10"
                >
                  {passwordVisible ? (
                    <FontAwesomeIcon icon={faEyeSlash} />
                  ) : (
                    <FontAwesomeIcon icon={faEye} />
                  )}
                </button>
                {formik.touched.password && formik.errors.password ? (
                  <div className="text-red-500">{formik.errors.password}</div>
                ) : null}
              </div>
              <div className="relative flex flex-col gap-2">
                <label>Confirm Password:</label>
                <input
                  type={confirmVisible ? "text" : "password"}
                  className={`outline-none py-2 pr-10 px-3 border border-grey rounded-xl ${
                    formik.touched.password_confirmation &&
                    formik.errors.password_confirmation
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                  placeholder="Enter Retype New Password"
                  name="password_confirmation"
                  value={formik.values.password_confirmation}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                />
                <button
                  type="button"
                  onClick={() => setConfirmVisible(!confirmVisible)}
                  className="absolute text-darkgrey right-3 top-10"
                >
                  {confirmVisible ? (
                    <FontAwesomeIcon icon={faEyeSlash} />
                  ) : (
                    <FontAwesomeIcon icon={faEye} />
                  )}
                </button>
                {formik.touched.password_confirmation &&
                formik.errors.password_confirmation ? (
                  <div className="text-red-500">
                    {formik.errors.password_confirmation}
                  </div>
                ) : null}
              </div>
              {error && <div className="text-red-500 text-center">{error}</div>}
              <button
                type="submit"
                className="bg-golden mt-10 text-white py-2 px-3 rounded-xl"
                disabled={formik.isSubmitting}
              >
                Sign up
              </button>
            </div>
          </form>
          <div className="flex flex-col gap-2 mt-8 md:mt-4 lg:mt-12">
            <div className="text-center">
              Already have an account?
              <Link href="/login" className="ml-2 text-golden underline">
                login
              </Link>
            </div>
          </div>
        </div>
      </div>
      <div></div>
      <div className="min-h-screen relative flex justify-end w-full md:w-2/3 lg:w-1/2">
        <Image src={bgimg} alt="bgimg" className="z-0 w-full min-h-full" />
        <div className="absolute hidden md:flex justify-center items-center h-full w-full">
          <Image
            src={image}
            alt="bgimg"
            className="lg:min-h-[360px] min-h-[300px] xl:max-w-[500px] lg:max-w-[300px] md:max-w-[300px]"
          />
        </div>
      </div>
    </div>
  );
};

export default Signup;
