import React from "react";
import * as Yup from "yup";
import { useFormik } from "formik";

const ExperienceModal = ({ isOpen, onClose, onSubmit, selectedExperience }) => {

  const formik = useFormik({
    initialValues: {
      company_name: selectedExperience?.company_name || "",
      designation: selectedExperience?.designation || "",
      start_date: selectedExperience?.start_date || "",
      end_date: selectedExperience?.end_date || "",
    },

    enableReinitialize: true,
    validationSchema: Yup.object({
      company_name: Yup.string().required("Institute Name is required"),
      designation: Yup.string().required("Field of Study is required"),
      start_date: Yup.date().required("Start Date is required"),
      end_date: Yup.date().required("End Date is required"),
    }),
    onSubmit: (values, { setSubmitting }) => {
      onSubmit(values);
      setSubmitting(false);
    },
  });

  if (!isOpen) {
    return null;
  }

  return (
    <div className="fixed inset-0 flex items-center justify-center bg-black bg-opacity-50">
      <div className="bg-[#FFFAFA] p-8 rounded-lg w-full max-w-lg">
        <form onSubmit={formik.handleSubmit}>
          <div className="flex flex-col gap-6">
            <div className="text-2xl font-bold">
              {selectedExperience ? "Edit Education" : "Add Education"}
            </div>

            <div className="flex flex-col">
              <label className="font-bold">Institute Name</label>
              <input
                type="text"
                name="company_name"
                value={formik.values.company_name}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                className={`outline-none bg-[#FFFAFA] border rounded-lg px-3 py-2 ${
                  formik.errors.company_name && formik.touched.company_name
                    ? "border-red-500"
                    : "border-gray-700 focus:border-golden"
                }`}
              />
              {formik.touched.company_name && formik.errors.company_name && (
                <div className="text-red-500 text-sm">
                  {formik.errors.company_name}
                </div>
              )}
            </div>

            <div className="flex flex-col">
              <label className="font-bold">Field of Study</label>
              <input
                type="text"
                name="designation"
                value={formik.values.designation}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                className={`outline-none bg-[#FFFAFA] border rounded-lg px-3 py-2 ${
                  formik.errors.designation && formik.touched.designation
                    ? "border-red-500"
                    : "border-gray-700 focus:border-golden"
                }`}
              />
              {formik.touched.designation && formik.errors.designation && (
                <div className="text-red-500 text-sm">
                  {formik.errors.designation}
                </div>
              )}
            </div>

            <div className="flex gap-4">
              <div className="flex flex-col w-1/2">
                <label className="font-bold">Start Date</label>
                <input
                  type="date"
                  name="start_date"
                  value={formik.values.start_date}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  className={`outline-none bg-[#FFFAFA] border rounded-lg px-3 py-2 ${
                    formik.errors.start_date && formik.touched.start_date
                      ? "border-red-500"
                      : "border-gray-700 focus:border-golden"
                  }`}
                />
                {formik.touched.start_date && formik.errors.start_date && (
                  <div className="text-red-500 text-sm">
                    {formik.errors.start_date}
                  </div>
                )}
              </div>

              <div className="flex flex-col w-1/2">
                <label className="font-bold">End Date</label>
                <input
                  type="date"
                  name="end_date"
                  value={formik.values.end_date}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  className={`outline-none bg-[#FFFAFA] border rounded-lg px-3 py-2 ${
                    formik.errors.end_date && formik.touched.end_date
                      ? "border-red-500"
                      : "border-gray-700 focus:border-golden"
                  }`}
                />
                {formik.touched.end_date && formik.errors.end_date && (
                  <div className="text-red-500 text-sm">
                    {formik.errors.end_date}
                  </div>
                )}
              </div>
            </div>
            <div className="flex justify-end gap-4">
              <button
                type="button"
                className="bg-gray-400 text-white px-4 py-2 rounded-lg"
                onClick={onClose}
              >
                Cancel
              </button>
              <button
                type="submit"
                className="bg-golden text-white px-4 py-2 rounded-lg"
              >
                {selectedExperience ? "Update" : "Add"}
              </button>
            </div>
          </div>
        </form>
      </div>
    </div>
  );
};

export default ExperienceModal;