"use client";
import React, { useEffect, useRef, useState } from "react";
import {
  DeleteRequest,
  GetRequest,
  PostRequest,
} from "../../../axiosConfig/axiosConfig";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import {
  faCirclePlay,
  faReply,
  faTrashCan,
} from "@fortawesome/free-solid-svg-icons";
import { faCircleCheck } from "@fortawesome/free-regular-svg-icons";
import ReplyModal from "./ReplyModal";
import toast from "react-hot-toast";
import { useRouter } from "next/navigation";
import { Spinner } from "@material-tailwind/react";
import {
  useCourseVideoMutation,
  useGetCommentsQuery,
  useGetCourseVideosQuery,
} from "@/services/Api";
import { useSelector } from "react-redux";
import { MdAssignmentAdd } from "react-icons/md";
import Notifications from "@/app/components/notifications/Notifications";
import SearchBox from "@/app/components/searchBox/SearchBox";

const CourseVideos = (props) => {

  const router = useRouter();
  const [subscriptionModal, setSubscriptionModal] = useState(false);
  const [enrollModal, setEnrollModal] = useState(false);
  const [enrollError, setEnrollError] = useState("");
  const [courseId, setCourseId] = useState("");
  const [commentText, setCommentText] = useState("");
  const [videoId, setVideoId] = useState(null);
  const [videoDetail, setVideoDetail] = useState({
    videoUrl: "",
    summary: "",
    content: "",
    title: "",
  });
  const [responsiveness, setResponsiveness] = useState({
    description: true,
    comments: false,
    upcomingVideos: false,
  });
  const videoRef = useRef(null);
  const [seeMore, setSeeMore] = useState(false);
  const [isModalOpen, setModalOpen] = useState(false);
  const [selectedComment, setSelectedComment] = useState(null);
  const { id } = props;
  const auth = useSelector((data) => data?.auth);

  const { data: getCourseVideos, refetch: getCourseVideosRefetch } =
    useGetCourseVideosQuery({ id: id });

  const handleNavigate = (id) => {
    router.push(`/fill-assignment?id=${id}`);
  };

  const handleNavigateToSubscribe = () => {
    router.push(`/payment?course=${id}`);
  };

  const { data: getComments, refetch: getCommentsRefetch } =
    useGetCommentsQuery({
      id: videoId || 1,
    });

  useEffect(() => {
    if (getCourseVideos?.course_videos?.length) {
      setVideoDetail({
        videoUrl:
          getCourseVideos?.course_videos[0]?.video?.media[0]?.original_url,
        summary: getCourseVideos?.course_videos[0]?.video?.summary,
        content: getCourseVideos?.course_videos[0]?.video?.content,
        title: getCourseVideos?.course_videos[0]?.video?.title,
      });

      setCourseId(getCourseVideos?.course_videos[0]?.course_id);
      setVideoId(getCourseVideos?.course_videos[0]?.id);
    }
  }, [getCourseVideos?.course_videos?.length]);

  const commentsDone = (status) => {
    if (status) {
      getCommentsRefetch();
    }
  };

  useEffect(() => {
    getCourseVideosRefetch();
    getCommentsRefetch();
  }, []);

  const [courseVideo] = useCourseVideoMutation();

  const logVideoViewApiHandler = (id) => {
    const data = {
      interaction_type: "video",
    };

    courseVideo({ data: data, id })
      .unwrap()
      .then(() => {
        getCourseVideosRefetch();
      })
      .catch((error) => { });
  };

  const handleVideoSelection = async (video) => {
    if (video.access === 0) {
      if (video.is_subscribe === 0) {
        setSubscriptionModal(true);
        return;
      }

      if (video.is_enroll === 0) {
        setEnrollModal(true);
        return;
      }
    }

    if (video.access === 0 && video.is_video_completed === 0) {
      logVideoViewApiHandler(video?.id);
    }

    const videoUrl = video?.video?.media?.[0]?.original_url;
    const videoDesc = video?.video?.summary;
    const videoCont = video?.video?.content;
    const videoTitle = video?.video?.title;

    setVideoId(video.id);
    if (videoUrl) {
      setVideoDetail({
        videoUrl: videoUrl || "No Video available.",
        summary: videoDesc || "No description available.",
        content: videoCont || "No content available.",
        title: videoTitle || "No title available.",
      });
    } else {
      console.error("Video is not available.");
    }
  };

  const openModal = (comment) => {
    setSelectedComment(comment);
    setModalOpen(true);
  };

  const closeModal = () => {
    setModalOpen(false);
    setSelectedComment(null);
  };

  const handleAddComments = async (e) => {
    e.preventDefault();
    const body = commentText;
    const data = {
      body,
    };
    try {
      const authToken = localStorage.getItem("authToken");
      const res = await PostRequest(
        `/course-videos/${videoId}/comments`,
        data,
        authToken
      );
      if (res) {
        toast.success("Comment Successful!", {
          position: "top-center",
        });
        setCommentText("");
      }
      try {
        const resComments = await GetRequest(
          `/course-videos/${videoId}/comments`,
          authToken
        );
        getCommentsRefetch();
      } catch (error) {
        console.error("Failed to fetch comments:", error);
        return;
      }
    } catch (error) {
      console.error("Reply error:", error.message);
      toast.error("Unable to  comment", {
        position: "top-right",
      });
    }
  };

  const handleEnroll = async (setEnrollModal) => {
    try {
      const authToken = localStorage.getItem("authToken");
      if (!authToken) {
        toast.error("Token not found.");
        return;
      }

      const response = await PostRequest(
        `/enroll/${courseId}/enroll`,
        {},
        authToken
      );

      if (response) {
        toast.success("You have successfully enrolled in the course.");
        setEnrollModal(false);
        getCourseVideosRefetch();
      }
    } catch (error) {
      console.error("Error enrolling in course:", error.response);
      toast.error("An error occurred while enrolling.", {
        position: "top-center",
      });
      setEnrollError(error.response.data.message);
    }
  };

  const handleTabClick = (tab) => {
    setResponsiveness({
      description: tab === "description",
      comments: tab === "comments",
      upcomingVideos: tab === "upcomingVideos",
    });
  };

  const handleDeleteComment = async (id) => {
    const authToken = localStorage.getItem("authToken");
    try {
      const res = await DeleteRequest(`/comment/${id}`, authToken);
      if (res) {
        toast.success("Comment deleted successfully.", {
          position: "top-center",
        });
        getCommentsRefetch();
      }
    } catch (error) {
      toast.error("Unable to delete comment", {
        position: "top-center",
      });
      console.log("Failed to delete comment:", error.response.data);
    }
  };

  const groupedVideos = [];
  let currentChapter = "";

  getCourseVideos?.course_videos?.forEach((video) => {
    if (video.chapter_name) {
      currentChapter = video.chapter_name;
    }

    if (!groupedVideos[currentChapter]) {
      groupedVideos[currentChapter] = [];
    }
    groupedVideos[currentChapter].push(video);
  });

  return (
    <div className="md:px-[28px] px-3 pb-8 min-h-screen">
      <div>
        {getCourseVideos?.course_videos?.length ? (
          <>
            <h1 className="py-2 text-[34px] font-bold pt-5">Courses</h1>
            <div className="flex flex-col sm:flex-row gap-4 sm:gap-0 items-end md:items-center sm:justify-between">
              <div className="flex w-full items-center justify-between gap-2">
                <div className="md:w-2/3">
                  <SearchBox />
                </div>
                <div className="flex items-center gap-2 text-darkgrey">
                  <Notifications />
                </div>
              </div>
            </div>
            <div className="md:bg-[#FFFFFF] min-h-[80vh] rounded-xl md:p-8 mt-6">
              <div className="flex flex-col md:flex-row md:gap-10">
                <div className="w-full">
                  <div className="bg-[#FFFFFF] md:p-0 p-5 rounded-xl">
                    <h1 className="font-bold lg:text-3xl md:text-2xl text-lg mb-4">
                      {videoDetail?.title}
                    </h1>
                    <div className="-z-10">
                      {videoDetail?.videoUrl ? (
                        <div>
                          <video
                            key={videoDetail?.videoUrl}
                            ref={videoRef}
                            width="714"
                            height="560"
                            autoPlay
                            playsInline
                            controls
                            preload="none"
                            className="w-full h-auto rounded-3xl -z-10"
                            onEnded={() => videoRef.current.pause()}
                          >
                            <source
                              src={videoDetail?.videoUrl}
                              type="video/mp4"
                            />
                            Your browser does not support the video tag.
                          </video>
                        </div>
                      ) : (
                        <p className="text-lg flex justify-center items-center">
                          <Spinner
                            color="info"
                            aria-label="Info spinner example"
                          />
                        </p>
                      )}
                    </div>
                  </div>
                  <div className="bg-[#FFFFFF] rounded-xl">
                    <div className="md:hidden border-b border-golden flex justify-evenly rounded-t-xl mt-7 bg-[#FFFFFF]">
                      <div
                        className={`text-lg cursor-pointer h-[100%] py-3 ${responsiveness?.description
                          ? "font-bold border-b-2 border-golden "
                          : ""
                          }`}
                        onClick={() => handleTabClick("description")}
                      >
                        Description
                      </div>
                      <div
                        className={`text-lg cursor-pointer h-[100%] py-3 ${responsiveness?.comments
                          ? "font-bold border-b-2 border-golden "
                          : ""
                          }`}
                        onClick={() => handleTabClick("comments")}
                      >
                        Comments
                      </div>
                      <div
                        className={`text-lg cursor-pointer h-[100%] py-3 ${responsiveness?.upcomingVideos
                          ? "font-bold border-b-2 border-golden "
                          : ""
                          }`}
                        onClick={() => handleTabClick("upcomingVideos")}
                      >
                        Videos
                      </div>
                    </div>
                    <div
                      className={`w-full mt-6 ${responsiveness?.description ? "block" : "hidden"
                        } md:block`}
                    >
                      <p className="md:block hidden text-secondary font-bold text-xl">
                        Description
                      </p>
                      <div className="bg-white md:py-4 p-4 rounded-2xl mt-4">
                        <p className={seeMore ? "" : ""}>
                          {seeMore ? (
                            <span
                              dangerouslySetInnerHTML={{
                                __html: videoDetail?.content,
                              }}
                            />
                          ) : (
                            videoDetail?.summary
                          )}
                        </p>
                        <button
                          className="font-semibold"
                          onClick={() => setSeeMore(!seeMore)}
                        >
                          {seeMore ? "See Less" : "See More"}
                        </button>
                      </div>
                    </div>
                    <div
                      className={`w-full mt-6 ${responsiveness?.comments ? "block" : "hidden"
                        } md:block`}
                    >
                      <p className="md:block hidden text-secondary font-bold text-xl my-6">
                        Comments
                      </p>
                      <div className="bg-white md:py-4 p-4 rounded-2xl mt-4">
                        <div>
                          <form onSubmit={handleAddComments}>
                            <div>
                              <p className="font-bold text-xl">Add Comment</p>
                              <textarea
                                value={commentText}
                                onChange={(e) => setCommentText(e.target.value)}
                                className=" bg-[#F9F5F5] my-2 w-full p-4 rounded-xl border border-gray-300 focus:outline-none focus:border-golden resize-none"
                                rows="4"
                                placeholder="Write your comment here..."
                              />
                            </div>
                            <button
                              type="submit"
                              className="w-full text-center bg-golden py-2 rounded-lg text-white text-sm font-bold"
                            >
                              Submit
                            </button>
                          </form>
                        </div>
                        <div className="mt-4 flex flex-col gap-3">
                          {getComments?.length ?
                            getComments?.map((comment) => (
                              <div
                                key={comment?.id}
                                className="flex flex-col gap-4 p-4 rounded-lg border border-gray-400"
                              >
                                <div className="flex flex-col gap-4">
                                  <div className="flex justify-between">
                                    <div className="flex items-center gap-2">
                                      <img
                                        className="w-8 h-8 rounded-full"
                                        src={
                                          comment?.user?.media[0]?.original_url
                                        }
                                      />
                                      <p className="font-bold text-sm">
                                        {comment?.user?.name}
                                      </p>
                                    </div>
                                    {comment?.user_id ==
                                      auth?.userDetail?.id ? (
                                      <button
                                        className="border border-golden p-1 bg-[#D8A32B7A] rounded-md text-white"
                                        onClick={() => {
                                          handleDeleteComment(comment?.id);
                                        }}
                                      >
                                        <FontAwesomeIcon
                                          icon={faTrashCan}
                                          className="h-4 w-4 text-[#D8A32B]"
                                        />
                                      </button>
                                    ) : null}
                                  </div>
                                  <div>
                                    <p className="text-secondary ml-8">
                                      {comment?.body}
                                    </p>
                                  </div>
                                  <div className="flex flex-col gap-3">
                                    {comment?.replies?.length &&
                                      comment?.replies?.map((reply) => (
                                        <div className="ml-10">
                                          <div className="flex flex-col gap-4">
                                            <div className="flex justify-between">
                                              <div className="flex items-center gap-2">
                                                <img
                                                  className="w-8 h-8 rounded-full"
                                                  src={
                                                    reply?.user.media[0]
                                                      ?.original_url
                                                  }
                                                />
                                                <p className=" font-bold text-sm">
                                                  {reply?.user?.name}
                                                </p>
                                              </div>
                                            </div>
                                            <p className="text-secondary ml-8">
                                              {reply?.body}
                                            </p>
                                          </div>
                                        </div>
                                      ))}
                                  </div>
                                  <div>
                                    <button
                                      className="text-golden flex items-center"
                                      onClick={() => openModal(comment)}
                                    >
                                      <FontAwesomeIcon
                                        icon={faReply}
                                        className="h-3 w-4"
                                      />
                                      Reply
                                    </button>
                                  </div>
                                </div>
                              </div>
                            )) : null}
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
                <div
                  className={`md:w-[300px] bg-[#FFFFFF] p-4 md:pt-0 ${responsiveness.upcomingVideos ? "block" : "hidden"
                    } md:block `}
                >
                  <h1 className="md:block hidden mb-3 text-xl font-bold text-secondary">
                    Upcoming Videos
                  </h1>

                  <div className="max-h-[700px] overflow-y-auto pr-2 flex flex-col gap-3 customScrollbar">
                    {Object.keys(groupedVideos).map((chapterName, index) => (
                      <div key={index} className="mb-4">
                        <h2 className="text-xl font-semibold mb-2">
                          {chapterName}
                        </h2>

                        {groupedVideos[chapterName].map((video, vidIndex) => (
                          <div
                            key={video?.id}
                            className={`relative mb-4 flex gap-4 px-2 rounded-lg py-3 border 
                              ${video?.access === 1 ||
                                video?.is_video_completed === 1
                                ? "border-green-500"
                                : "border-gray-400"
                              }
                              ${!video?.parent_completed
                                ? "opacity-50 pointer-events-none"
                                : "cursor-pointer"
                              }
                            `}
                            onClick={() => {
                              handleVideoSelection(video, vidIndex);
                            }}
                          >
                            {video?.areAllParentsCompleted &&
                              video?.assignment &&
                              video?.is_enroll == 1 &&
                              video?.is_subscribe == 1 && (
                                <div
                                  className="absolute top-1 right-1 h-4 w-4 cursor-auto"
                                  onClick={() =>
                                    handleNavigate(video?.assignment?.id)
                                  }
                                >
                                  <MdAssignmentAdd className="text-golden" />
                                </div>
                              )}

                            <div className="flex items-center gap-2 pt-2">
                              {video?.access === 1 ||
                                video?.is_video_completed === 1 ? (
                                <FontAwesomeIcon
                                  icon={faCircleCheck}
                                  className="h-4 w-4 text-green-500"
                                />
                              ) : (
                                <FontAwesomeIcon
                                  icon={faCirclePlay}
                                  className="h-4 w-4"
                                />
                              )}
                              <div>
                                <div className="text-sm font-normal text-secondary">
                                  {video?.video?.title}
                                </div>
                              </div>
                            </div>
                          </div>
                        ))}
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            </div>
            <ReplyModal
              isOpen={isModalOpen}
              onClose={closeModal}
              comment={selectedComment}
              videoId={videoId}
              commented={commentsDone}
            />
          </>
        ) : (
          <>
            <div className="min-h-[90vh] flex justify-center items-center">
              <Spinner color="warning" aria-label="Warning spinner example" />
            </div>
          </>
        )}
      </div>

      {subscriptionModal ? (
        <div className="fixed inset-0 flex items-center justify-center bg-black bg-opacity-50">
          <div className="bg-white flex flex-col items-center p-6 rounded-lg shadow-lg max-w-sm w-full">
            <p className="mb-4 flex flex-col items-center">
              Subscribe to watch more videos
              <span onClick={handleNavigateToSubscribe} className="text-blue-500 underline cursor-pointer">
                Click to subscribe
              </span>
            </p>
            <button
              onClick={() => setSubscriptionModal(false)}
              className="bg-red-500 text-center text-white px-4 py-2 rounded hover:bg-red-600"
            >
              Close
            </button>
          </div>
        </div>
      ) : null}

      {enrollModal ? (
        <div className="fixed inset-0 flex items-center justify-center bg-black bg-opacity-70">
          <div className="bg-white flex flex-col items-center p-6 rounded-lg shadow-lg max-w-md w-full">
            <p className="mb-4 text-xl font-medium text-golden flex flex-col items-center">
              Enroll course to watch more videos
            </p>
            {enrollError && (
              <div className="text-red-500 text-base">{enrollError}</div>
            )}
            <div className="flex gap-4">
              <button
                onClick={() => setEnrollModal(false)}
                className="bg-gray-500 text-center text-white px-4 py-2 rounded hover:bg-gray-500"
              >
                Close
              </button>
              <button
                onClick={() => {
                  handleEnroll(setEnrollModal);
                }}
                className="bg-golden text-center text-white px-4 py-2 rounded"
              >
                Enroll
              </button>
            </div>
          </div>
        </div>
      ) : null}
    </div>
  );
};

export default CourseVideos;