"use client";
import React, { useEffect, useState } from "react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faPen, faTrash } from "@fortawesome/free-solid-svg-icons";
import { useFormik } from "formik";
import * as Yup from "yup";
import toast from "react-hot-toast";
import {
  DeleteRequest,
  PostRequest,
} from "@/app/components/axiosConfig/axiosConfig";
import EducationModal from "../EducationModal";
import { useGetEducationQuery } from "@/services/Api";
import { useRouter } from "next/navigation";

const Education = () => {

  const [loading, setLoading] = useState(false);
  const [allEducation, setAllEducation] = useState([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedEducation, setSelectedEducation] = useState(null);
  const router = useRouter();

  const {
    data: getEducation,
    refetch: getEducationRefetch,
  } = useGetEducationQuery();

  useEffect(() => {
    setAllEducation(getEducation?.education);
  }, [getEducation]);

  useEffect(() => {
    getEducationRefetch();
  }, []);

  const formik = useFormik({
    initialValues: {
      institute_name: "",
      field_of_study: "",
      start_date: "",
      end_date: "",
    },
    validationSchema: Yup.object({
      institute_name: Yup.string().required("Institute Name is required"),
      field_of_study: Yup.string().required("Field of Study is required"),
      start_date: Yup.date().required("Start Date is required"),
      end_date: Yup.date().required("End Date is required"),
    }),
    onSubmit: async (values, { setSubmitting, resetForm }) => {
      const authToken = localStorage.getItem("authToken");

      try {
        const res = await PostRequest("/education", values, authToken);
        toast.success("Education added successfully!", {
          position: "top-center",
        });
        getEducationRefetch();
        resetForm();
        router.push("/profile");
      } catch (error) {
        console.error("Update education Error", error);
        toast.error("Education Not Added!", {
          position: "top-center",
        });
      } finally {
        setLoading(false);
        setSubmitting(false);
      }
    },
  });

  const handleEditEducation = (education) => {
    setSelectedEducation(education);
    setIsModalOpen(true);
  };

  const handleDeleteEducation = async (id) => {
    const confirmDelete = window.confirm(
      "Are you sure you want to delete this Education?"
    );

    if (confirmDelete) {
      const authToken = localStorage.getItem("authToken");
      try {
        const res = await DeleteRequest(`/education/${id}`, authToken);

        toast.success("Education deleted successfully!", {
          position: "top-center",
        });

        getEducationRefetch();
      } catch (error) {
        console.error("Failed to delete Education:", error);
        toast.error("Failed to delete Education", {
          position: "top-center",
        });
      }
    }
  };

  const handleModalSubmit = async (updatedEducation) => {
    setLoading(true);
    const authToken = localStorage.getItem("authToken");
    try {
      await PostRequest(
        `/education/${selectedEducation.id}`,
        updatedEducation,
        authToken
      );
      toast.success("Education updated successfully!", {
        position: "top-center",
      });
      getEducationRefetch();
      setIsModalOpen(false);
    } catch (error) {
      console.error("Failed to update Education:", error);
      toast.error("Failed to update Education.", { position: "top-center" });
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen md:px-10 px-4 py-6">
      <div className="bg-[#FFFAFA] min-h-[95vh] rounded-2xl p-4 lg:p-8">
        <div className="text-3xl font-bold">Education</div>
        <div className="md:w-2/3 mt-10">
          {allEducation?.length ? (
            <div>
              {allEducation?.map((item, index) => (
                <div key={index} className="flex flex-col gap-4 mb-8">
                  <div className="flex justify-between items-center">
                    <div className="text-2xl text-golden font-bold">
                      Experience
                    </div>
                    <div className="flex gap-4">
                      <button
                        onClick={() => handleDeleteEducation(item?.id)}
                      >
                        <FontAwesomeIcon
                          icon={faTrash}
                          className="text-golden w-4 h-4"
                        />
                      </button>
                      <button
                        onClick={() => handleEditEducation(item)}
                      >
                        <FontAwesomeIcon
                          icon={faPen}
                          className="text-golden w-4 h-4"
                        />
                      </button>
                    </div>
                  </div>
                  <div className="flex gap-2 justify-between">
                    <div className="flex gap-1 flex-col items-start">
                      <span className="text-lg font-bold">Institute Name</span>
                      <span className="text-base text-[#0000005C] font-medium">
                        {item?.institute_name}
                      </span>
                    </div>
                    <div className="min-w-36 flex gap-1 items-start flex-col">
                      <span className="text-lg font-bold">Field of Study</span>
                      <span className="text-base text-[#0000005C] font-medium">
                        {item?.field_of_study}
                      </span>
                    </div>
                  </div>
                  <div className="flex gap-2 justify-between">
                    <div className="flex gap-1 items-start flex-col">
                      <span className="text-lg font-bold">Start Date</span>
                      <span className="text-base text-[#0000005C] font-medium">
                        {item?.start_date}
                      </span>
                    </div>
                    <div className="min-w-36 flex gap-1 items-start flex-col">
                      <span className="text-lg font-bold">End Date</span>
                      <span className="text-base text-[#0000005C] font-medium">
                        {item?.end_date}
                      </span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          ) : (
            <div>No education data available.</div>
          )}
        </div>
        <div className="mt-8">
          <form onSubmit={formik.handleSubmit}>
            <div className="flex flex-col-reverse md:flex-row gap-6 justify-between">
              <div className="md:w-2/3 flex flex-col gap-6">
                <div className="text-2xl text-golden font-bold">
                  Add New Education
                </div>

                <div className="flex gap-2 flex-col">
                  <label className="text-[16px] font-bold">
                    Institute Name
                  </label>
                  <input
                    type="text"
                    name="institute_name"
                    value={formik.values.institute_name}
                    onChange={formik.handleChange}
                    onBlur={formik.handleBlur}
                    placeholder="Enter Institute Name"
                    className={`outline-none border rounded-lg px-3 py-3 bg-[#FFFAFA] ${formik.touched.institute_name &&
                      formik.errors.institute_name
                      ? "border-red-500"
                      : "border-gray-700 focus:border-golden"
                      }`}
                  />
                  {formik.touched.institute_name &&
                    formik.errors.institute_name ? (
                    <div className="text-red-500 text-sm">
                      {formik.errors.institute_name}
                    </div>
                  ) : null}
                </div>

                <div className="flex gap-2 flex-col">
                  <label className="text-[16px] font-bold">
                    Field of Study
                  </label>
                  <input
                    type="text"
                    name="field_of_study"
                    value={formik.values.field_of_study}
                    onChange={formik.handleChange}
                    onBlur={formik.handleBlur}
                    placeholder="Enter Field of Study"
                    className={`outline-none border rounded-lg px-3 py-3 bg-[#FFFAFA] ${formik.touched.field_of_study &&
                      formik.errors.field_of_study
                      ? "border-red-500"
                      : "border-gray-700 focus:border-golden"
                      }`}
                  />
                  {formik.touched.field_of_study &&
                    formik.errors.field_of_study ? (
                    <div className="text-red-500 text-sm">
                      {formik.errors.field_of_study}
                    </div>
                  ) : null}
                </div>

                <div className="flex gap-4">
                  <div className="flex gap-2 flex-col w-1/2">
                    <label className="text-[16px] font-bold">Start Date</label>
                    <input
                      type="date"
                      name="start_date"
                      value={formik.values.start_date}
                      onChange={formik.handleChange}
                      onBlur={formik.handleBlur}
                      placeholder="Enter Start Date"
                      className={`outline-none border rounded-lg px-3 py-3 bg-[#FFFAFA] ${formik.touched.start_date && formik.errors.start_date
                        ? "border-red-500"
                        : "border-gray-700 focus:border-golden"
                        }`}
                    />
                    {formik.touched.start_date && formik.errors.start_date ? (
                      <div className="text-red-500 text-sm">
                        {formik.errors.start_date}
                      </div>
                    ) : null}
                  </div>
                  <div className="flex gap-2 flex-col w-1/2">
                    <label className="text-[16px] font-bold">End Date</label>
                    <input
                      type="date"
                      name="end_date"
                      value={formik.values.end_date}
                      onChange={formik.handleChange}
                      onBlur={formik.handleBlur}
                      placeholder="Enter End Date"
                      className={`outline-none border rounded-lg px-3 py-3 bg-[#FFFAFA] ${formik.touched.end_date && formik.errors.end_date
                        ? "border-red-500"
                        : "border-gray-700 focus:border-golden"
                        }`}
                    />
                    {formik.touched.end_date && formik.errors.end_date ? (
                      <div className="text-red-500 text-sm">
                        {formik.errors.end_date}
                      </div>
                    ) : null}
                  </div>
                </div>
                <div className="flex justify-end mt-6">
                  <button
                    type="submit"
                    className=" bg-golden rounded-xl px-6 text-white font-medium py-2"
                    disabled={loading}
                  >
                    {loading ? "loading..." : "Add Education"}
                  </button>
                </div>
              </div>
            </div>
          </form>
        </div>
      </div>
      {isModalOpen && (
        <div>
          <EducationModal
            isOpen={isModalOpen}
            onClose={() => setIsModalOpen(false)}
            onSubmit={handleModalSubmit}
            selectedEducation={selectedEducation}
          />
        </div>
      )}
    </div>
  );
};

export default Education;