"use client";
import React, { useEffect, useState } from "react";
import toast from "react-hot-toast";
import {
  useAddCoupinMutation,
  useAddSubscriptionMutation,
  useCheckSubscriptionQuery,
  useDelSubscriptionMutation,
  useGetPaymentIntentQuery,
  useGetPlansQuery,
} from "@/services/Api";
import { loadStripe } from "@stripe/stripe-js";
import {
  Elements,
  useStripe,
  useElements,
  CardNumberElement,
  CardExpiryElement,
  CardCvcElement,
} from "@stripe/react-stripe-js";
import { useRouter, useSearchParams } from "next/navigation";
import Header from "../header/header";

const CheckoutForm = () => {
  const [amount, setAmount] = useState("");
  const [discountedAmount, setDiscountedAmount] = useState("");
  const [coupinId, setCoupinId] = useState("");
  const [planId, setPlanId] = useState("");
  const [couponCode, setCouponCode] = useState("");
  const router = useRouter();
  const stripe = useStripe();
  const elements = useElements();
  const searchParams = useSearchParams();
  const courseId = searchParams.get("course");

  const {
    data: getPlans,
    refetch: getPlansRefetch,
    isLoading: getPlansIsLoading,
  } = useGetPlansQuery();

  const { data: checkSubscription, refetch: checkSubscriptionRefetch } =
    useCheckSubscriptionQuery();

  const [addCoupin, { isLoading: addCoupinLoading }] = useAddCoupinMutation();
  const [addSubscription, { isLoading: addSubscriptionLoading }] =
    useAddSubscriptionMutation();
  const [delSubscription, { isLoading: delSubscriptionLoading }] =
    useDelSubscriptionMutation();

  const handlePlanChange = (plan) => {
    setAmount(plan?.amount);
    setPlanId(plan?.id);
    setCouponCode("");
  };

  const addCoupinApi = () => {
    const data = {
      code: couponCode,
      plan_id: planId,
    };

    addCoupin({ data: data })
      .unwrap()
      .then((payload) => {
        setDiscountedAmount(payload?.after_discounted_price);
        setCoupinId(payload?.coupon_details?.id);
        setCouponCode("");
        toast.success(payload?.message, {
          position: "top-center",
        });
      })
      .catch((err) => {
        toast.error("Error...!!!", {
          position: "top-center",
        });
        toast.error(err?.data?.message, {
          position: "top-center",
        });
      });
  };

  const addSubscriptionApi = () => {
    if (!stripe || !elements) {
      toast.error("Stripe has not loaded yet. Please try again later.");
      return;
    }

    const data = {
      plan_id: planId,
      coupon_id: coupinId,
      add_more_subs: true,
    };

    const cardNumberElement = elements.getElement(CardNumberElement);

    stripe
      .createToken(cardNumberElement)
      .then((result) => {
        if (result.error) {
          toast.error(result.error.message, {
            position: "top-center",
          });
          return;
        }

        data.token = result.token;

        addSubscription({ data })
          .unwrap()
          .then(() => {
            if (courseId) {
              router.push(`/courses/${courseId}`);
            } else {
              window.location.reload();
            }
          })
          .catch(() => {
            toast.error("Error adding subscription. Please try again.", {
              position: "top-center",
            });
          });
      })
      .catch((error) => {
        console.error("Token creation error:", error);
        toast.error("An unexpected error occurred. Please try again.", {
          position: "top-center",
        });
      });
  };

  const delSubscriptionApi = () => {
    delSubscription()
      .unwrap()
      .then(() => {
        checkSubscriptionRefetch();
      })
      .catch(() => {
        toast.error("Error...!!!", {
          position: "top-center",
        });
      });
  };

  useEffect(() => {
    checkSubscriptionRefetch();
    getPlansRefetch();
  }, []);

  const cardElementOptions = {
    style: {
      base: {
        color: "#32325d",
        fontFamily: "Arial, sans-serif",
        fontSmoothing: "antialiased",
        fontSize: "16px",
        "::placeholder": {
          color: "#aab7c4",
        },
      },
      invalid: {
        color: "#fa755a",
        iconColor: "#fa755a",
      },
    },
  };

  return (
    <>
      <div className="md:px-[28px] px-3 pb-8 min-h-screen">
        <Header />
        <div className="min-h-[80vh] flex items-center justify-center">
            <div className="bg-white shadow-lg rounded-lg pt-10 pb-20 w-full text-center md:px-4 px-4 xl:max-w-[1040px] xl:min-w-[900px] xl:text-left">
              <h1 className="text-[34px] font-bold mb-8">Membership</h1>

              {!checkSubscription?.subscription ? (
                <div className="grid md:grid-cols-2 gap-x-4">
                  <div className="mb-6">
                    <h2 className="text-2xl font-bold mb-6">Select Plan</h2>
                    <div className="px-2">
                      <div className="bg-[#ACACAC] h-[1px]"></div>
                    </div>
                    <div className="flex flex-col gap-2 pt-8">
                      {getPlansIsLoading ? (
                        <div className="text-center">Loading...</div>
                      ) : getPlans?.length ? (
                        getPlans?.map((plan) => (
                          <label
                            key={plan.id}
                            className={`flex items-center p-2 rounded-lg mb-2 bg-[#FDFDFD] border ${
                              amount == plan?.amount
                                ? "border-blue-400"
                                : "border-transparent"
                            }`}
                          >
                            <input
                              type="radio"
                              name="plan"
                              value={plan?.amount}
                              className="mr-2"
                              onChange={() => handlePlanChange(plan)}
                            />
                            <div className="flex flex-col pl-2">
                              <label className="text-[#1B252B] font-semibold">
                                {plan?.title}
                              </label>
                              <label>
                                {plan?.amount} /{" "}
                                <span className="capitalize">
                                  {plan?.interval}
                                </span>
                              </label>
                            </div>
                          </label>
                        ))
                      ) : (
                        <div className="text-center text-secondary font-semibold text-2xl">
                          No data available
                        </div>
                      )}
                    </div>

                    {getPlans?.length ? (
                      <div>
                        <input
                          type="text"
                          className="mt-4 mb-4 outline-none py-2 px-3 border border-grey rounded-xl"
                          placeholder="Coupon code"
                          value={couponCode}
                          onChange={(e) => setCouponCode(e.target.value)}
                        />

                        <button
                          className="bg-golden mt-10 text-white py-2 px-3 rounded-xl ms-5"
                          onClick={addCoupinApi}
                          disabled={addCoupinLoading || !couponCode}
                        >
                          {addCoupinLoading ? "Loading..." : "Apply"}
                        </button>

                        <h3 className="mb-2">
                          <b>
                            {discountedAmount
                              ? `Origional Price: $${amount}`
                              : ""}
                          </b>
                        </h3>
                        <h3>
                          <b>
                            {discountedAmount
                              ? `Discounted Price: $${discountedAmount}`
                              : amount
                              ? `Total Price: $${amount}`
                              : ""}
                          </b>
                        </h3>
                      </div>
                    ) : null}
                  </div>

                  {getPlans?.length ? (
                    <div>
                      <div className="mb-6">
                        <h2 className="text-2xl font-bold mb-6">
                          Payment With Card
                        </h2>
                        <div className="px-2">
                          <div className="bg-[#ACACAC] h-[1px]"></div>
                        </div>

                        <div className="mt-7 mb-10 cardElem">
                          <label>Card Number</label>
                          <div className="mt-2">
                            <CardNumberElement options={cardElementOptions} />
                          </div>

                          <div className="grid md:grid-cols-2 gap-x-4 mt-5">
                            <div>
                              <label>Expiration Date</label>
                              <div className="mt-2">
                                <CardExpiryElement
                                  options={cardElementOptions}
                                />
                              </div>
                            </div>

                            <div>
                              <label>CVC</label>
                              <div className="mt-2">
                                <CardCvcElement options={cardElementOptions} />
                              </div>
                            </div>
                          </div>
                        </div>

                        <div className="text-center mt-3">
                          <button
                            disabled={
                              addSubscriptionLoading || !stripe || !elements
                            }
                            className="w-full bg-blue-600 text-white py-2 rounded-lg font-bold hover:bg-blue-700"
                            onClick={addSubscriptionApi}
                          >
                            {addSubscriptionLoading ? "Loading..." : "Pay"}{" "}
                            {discountedAmount
                              ? `$${discountedAmount}`
                              : amount
                              ? `$${amount}`
                              : ""}
                          </button>
                        </div>
                      </div>
                      <div className="text-[#ACACAC] text-sm">
                        Your personal data will be used to process your order,
                        support your experience throughout this website, and for
                        other purposes described in our privacy policy.
                      </div>
                    </div>
                  ) : null}
                </div>
              ) : (
                <>
                  <h3 className="mb-3">
                    <b>Start Date:</b>{" "}
                    {checkSubscription?.subscription?.start_date}
                  </h3>
                  <h3 className="mb-3">
                    <b>End Date:</b> {checkSubscription?.subscription?.end_date}
                  </h3>
                  <h3 className="mb-3">
                    <b>Amount:</b> $
                    {checkSubscription?.subscription?.final_amount}
                  </h3>
                  <button
                    className="bg-red-500 mt-10 text-white py-2 px-3 rounded-xl"
                    onClick={delSubscriptionApi}
                    disabled={delSubscriptionLoading}
                  >
                    {delSubscriptionLoading
                      ? "Loading..."
                      : "Delete Subscription"}
                  </button>
                </>
              )}
            </div>
        </div>
      </div>
    </>
  );
};

const STRIPE_KEY =
  "pk_test_51PkU3BAzPkjS5j4VioBBaVZUYE5T4ECs12VYLvGCuJjK4ktwzEtuVkuuTheHSaRTK6CsEgMHjMDAyTi2opDLApoE000KwJQSq9";

const stripePromise = loadStripe(STRIPE_KEY);

const StripePaymentForm = () => {
  const { data: getPaymentIntent } = useGetPaymentIntentQuery();

  const appearance = {
    theme: "stripe",
  };

  const options = {
    clientSecret: getPaymentIntent?.intent?.client_secret,
    appearance,
  };

  return (
    <>
      <Elements options={options} stripe={stripePromise}>
        <CheckoutForm />
      </Elements>
    </>
  );
};

export default StripePaymentForm;
