"use client"; // Required for App Router if using client-side features

import React, { useState } from "react";
import Link from "next/link";
import Image from "next/image";
import Logo from "../../../public/assets/agent-ambition.png";
import { Formik, Form, Field, ErrorMessage } from "formik";
import * as Yup from "yup";
import { BASE_URL } from "@/services/ApiEndpoints";
import { useSelector } from "react-redux";
import PhoneInputWithCountrySelect from "react-phone-number-input";
import "react-phone-number-input/style.css"; // Import styles for phone input

// Yup validation schema
const validationSchema = Yup.object({
  name: Yup.string().trim().required("Name is required"),
  email: Yup.string()
    .email("Invalid email format")
    .required("Email is required"),
  phone: Yup.string()
    .matches(/^\+?\d{10,15}$/, "Invalid phone number (10-15 digits)")
    .required("Phone number is required"),
  message: Yup.string().optional(), // Message is optional
  consent: Yup.boolean()
    .oneOf([true], "You must agree to the privacy policy")
    .required("Consent is required"),
});

const ComingSoon = () => {
  const auth = useSelector((state) => state.auth); // Optional: for token
  const [responseMessage, setResponseMessage] = useState(""); // For API response

  // Initial form values
  const initialValues = {
    name: "",
    email: "",
    phone: "",
    message: "",
    consent: false,
  };

  // Handle form submission
  const handleSubmit = async (values, { setSubmitting, resetForm }) => {
    setResponseMessage("");

    try {
      const res = await fetch(BASE_URL + "coming-soon", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(values),
      });

      const data = await res.json();

      if (data.status) {
        setResponseMessage("Thank you! Your submission was successful.");
        resetForm();
      } else {
        setResponseMessage(
          data.error || "Something went wrong. Please try again."
        );
      }
    } catch (error) {
      setResponseMessage("Failed to submit the form. Please try again later.");
    } finally {
      setSubmitting(false);
    }
  };

  return (
    <div className="bg-[#000] w-full min-h-[100vh] coming-soon-page px-3">
      <div className="header-logo py-6 max-w-[1200px] lg:px-4 mx-auto relative z-2">
        <Link href="/" className="max-w-[120px] md:max-w-[180px] mx-auto block">
          <Image
            src={Logo}
            alt="Agent Ambition Logo"
            className="max-w-[120px] md:max-w-[180px] mx-auto"
          />
        </Link>
      </div>
      <div className="py-[40px] md:py-[60px] max-w-[1200px] lg:px-4 mx-auto relative z-2">
        <div className="text-center mb-6">
          <h1 className="text-[#b1b3b2] text-2xl md:text-4xl mb-[8px] font-semibold">
            We are coming soon!
          </h1>
          <p className="text-[#b1b3b2] text-sm md:text-xl">
            Stay tuned for something amazing
          </p>
        </div>
        <Formik
          initialValues={initialValues}
          validationSchema={validationSchema}
          onSubmit={handleSubmit}
        >
          {({ isSubmitting, errors, touched, setFieldValue }) => (
            <Form noValidate>
              <div className="col-span-full mb-[24px]">
                <label
                  htmlFor="name"
                  className="block font-medium text-[#b1b3b2] mb-[4px]"
                >
                  Name
                </label>
                <Field
                  type="text"
                  name="name"
                  id="name"
                  className={`w-full py-1.5 pr-3 px-2 h-[45px] rounded-md text-base text-white bg-transparent border ${
                    errors.name && touched.name
                      ? "border-red-500"
                      : "border-[#fff]"
                  } placeholder:text-gray-400 focus:outline-none`}
                  placeholder="Name"
                />
                <ErrorMessage
                  name="name"
                  component="p"
                  className="text-red-500 text-sm mt-1"
                />
              </div>
              <div className="col-span-full mb-[24px]">
                <label
                  htmlFor="email"
                  className="block font-medium text-[#b1b3b2] mb-[4px]"
                >
                  Email
                </label>
                <Field
                  type="email"
                  name="email"
                  id="email"
                  className={`w-full py-1.5 pr-3 px-2 h-[45px] rounded-md text-base text-white bg-transparent border ${
                    errors.email && touched.email
                      ? "border-red-500"
                      : "border-[#fff]"
                  } placeholder:text-gray-400 focus:outline-none`}
                  placeholder="Email"
                />
                <ErrorMessage
                  name="email"
                  component="p"
                  className="text-red-500 text-sm mt-1"
                />
              </div>
              <div className="col-span-full mb-[24px]">
                <label
                  htmlFor="phone"
                  className="block font-medium text-[#b1b3b2] mb-[4px]"
                >
                  Phone
                </label>
                <PhoneInputWithCountrySelect
                  international
                  countryCallingCodeEditable={false}
                  defaultCountry="US"
                  value={initialValues.phone}
                  onChange={(value) => setFieldValue("phone", value)}
                  className={`w-full py-1.5 pr-3 px-2 h-[45px] rounded-md text-base text-white bg-transparent border outline-none focus:outline-none ${
                    errors.phone && touched.phone
                      ? "border-red-500"
                      : "border-[#fff]"
                  } placeholder:text-gray-400`}
                  placeholder="Enter phone number"
                />
                <ErrorMessage
                  name="phone"
                  component="p"
                  className="text-red-500 text-sm mt-1"
                />
              </div>
              <div className="col-span-full mb-[8px]">
                <label
                  htmlFor="message"
                  className="block font-medium text-[#b1b3b2] mb-[4px]"
                >
                  Message
                </label>
                <Field
                  as="textarea"
                  name="message"
                  id="message"
                  rows="3"
                  className={`w-full py-1.5 pr-3 px-2 rounded-md text-base text-white bg-transparent border ${
                    errors.message && touched.message
                      ? "border-red-500"
                      : "border-[#fff]"
                  } placeholder:text-gray-400 focus:outline-none sm:text-sm/6`}
                />
                <ErrorMessage
                  name="message"
                  component="p"
                  className="text-red-500 text-sm mt-1"
                />
              </div>
              <div className="flex space-x-2 mb-[24px]">
                <Field
                  type="checkbox"
                  name="consent"
                  id="consent"
                  className={`w-4 h-4 mt-1 text-blue-600 border-gray-300 rounded focus:ring-blue-500 ${
                    errors.consent && touched.consent ? "border-red-500" : ""
                  }`}
                />
                <label htmlFor="consent" className="text-[#b1b3b2]">
                  By submitting this form, you consent to the processing of your
                  personal data and agree to be contacted via email and phone in
                  accordance with our{" "}
                  <Link href="/privacy-policy" className="underline">
                    Privacy Policy
                  </Link>
                  .
                </label>
              </div>
              <ErrorMessage
                name="consent"
                component="p"
                className="text-red-500 text-sm mb-2"
              />
              {responseMessage && (
                <p
                  className={`text-sm mb-4 ${
                    responseMessage.includes("Thank you")
                      ? "text-green-500"
                      : "text-red-500"
                  }`}
                >
                  {responseMessage}
                </p>
              )}
              <div className="text-center">
                <button
                  type="submit"
                  className="bg-golden text-white py-2 px-6 rounded-md text-[16px] font-medium disabled:opacity-50"
                  disabled={isSubmitting}
                >
                  {isSubmitting ? "Submitting..." : "Submit"}
                </button>
              </div>
            </Form>
          )}
        </Formik>
      </div>
    </div>
  );
};

export default ComingSoon;
